/* =+== 1 ===+=== 2 ===+=== 3 ===+=== 4 ===+=== 5 ===+=== 6 ===+=== 7 ===
Programm XPLOT.BAS fuer MOBILE BASIC unter ANDROID, Format FULL-HD. 
Datei steuert Plotter (Tablet 1920 x 1080 pixel) - 28jun14 01:00 jgt
========+=========+=========+=========+=========+=========+=========+====

Es wird zunaechst eine Plot-Datei "plot.dat" erzeugt, dann ausgelesen
und ausgefuehrt. Diese Plot-Datei koennte auch mit Gforth erzeugt sein.

    #############################################
    ##   G R A F I K   F U E R   G F O R T H   ##
    #############################################

Da Gforth auch unter Android laeuft, wird die Datei "xplot.dat" passend
im Ordner "/storage/emulated/0/gforth/home/" abgelegt und ausgelesen.

Alle Plot-Daten sind vom Typ "short" (16 bit), Text als Doppelbyte.
Die Kommandos werden durch Kennnummern ($AAxx) eingeleitet.

COLOR   | $AA01 | <red> | <gre> | <blu> | <alp> |
CLEAR   | $AA02 |
LINE    | $AA03 |  <x1> |  <y1> |  <x2> |  <y2> |
RECT    | $AA04 |  <x>  |  <y>  | <wid> | <hig> |  <f>  |
OVAL    | $AA05 |  <x>  |  <y>  | <wid> | <hig> |  <f>  |
CIRCLE  | $AA06 |  <x>  |  <y>  | <rad> |  <f>  |
PLOT    | $AA07 |  <x1> |  <y1> |  <x2> |  <y2> | . . . | . . . | $AA00 |
PAUSE   | $AA08 | <sec> |
SHOW    | $AA09 |
END     | $AA0A |
ARC     | $AA0B |  <x>  |  <y>  | <wid> | <hig> | <beg> | <arc> |  <f>  |
PIE     | $AA0C |  <x>  |  <y>  | <wid> | <hig> | <beg> | <arc> |  <f>  |
FONT    | $AA0D | <typ> | <hig> |
TEXT    | $AA0E |  <x>  |  <y>  | a   b | c   d | . . . | $AA00 |
ORIGIN  | $AA0F |  <x>  |  <y>  |
ROTATE  | $AA10 | <arc> |
RNDRECT | $AA11 |  <x>  |  <y>  | <wid> | <hig> | <rad> |  <f>  |
TRIANG  | $AA12 |  <x1> |  <y1> |  <x2> |  <y2> |  <x3> |  <y3> |  <f>  |
SAVE    | $AA13 |
RESTORE | $AA14 | 

COLOR:    <red><grn><blu>, <alpha> = (0 = transparent, 255 = deckend)
RECT:     <x><y> = Start, <wid><hig> = Breite, Hoehe; <f> = fuellen
OVAL:     wie RECT, unsichtbares Rechteck umschliesst Ellse
CIRCLE:   <x><y> = Mittelpunkt, <rad> = Radius; <f> = fuellen
ARC:      wie OVAL, dazu <beg><arc> = Startwinkel, Bogenlaenge
PIE:      wie ARC, aber Bogenenden mit Mittelpunkt verbunden
FONT:     <typ><hig> = (1=Mono, 2=SansSerif, 3=Serif), <hig> = Pixelhoehe
TEXT:     <x><y> = Start, ab cd ... = Doppel-Zeichen (big endian)
ORIGIN:   <x><y> neuer Bezugspunkt, alten mit SAVE/RESTORE bewahren
ROTATE:   <arc> = Winkelgrade (relativ), drehen um ORIGIN
RNDRECT:  <wid><hig> = Breite, Hoehe; <rad> = Eckradius; <f> = fuellen
TRIANG:   Dreieck <x1><y1><x2><y2><x3><y3>, mit <f> fuellen
*/

// Kommando setzen (Kommandonummer x wird zu $AA00+x ergaenzt)
sub wrcmd(x as integer)
    put #1,short(0xAA00+x) 
end sub

// Einen Parameter setzen
sub wrpar(x as integer)
    put #1,short(x)
end sub

// Zwei Parameter setzen (z.B. x y)
sub w2par(x as integer, y as integer)
    put #1,short(x)
    put #1,short(y)
end sub

// Plot anzeigen und 1 Sekunde warten
sub show
    call wrcmd(9)               // [9] SHOW
    call wrcmd(8)               // [8] PAUSE s
    call wrpar(1)
end sub

// Schwarze Farbe waehlen
sub black
    call wrcmd(1)               // [1] COLOR r g b alpha
    call wrpar(0)
    call wrpar(0)
    call wrpar(0)
    call wrpar(255)             // 0=transparent, 256=deckend
end sub

// Graue Farbe waehlen
sub grey
    call wrcmd(1)               // [1] COLOR r g b alpha
    call wrpar(100)
    call wrpar(100)
    call wrpar(100)
    call wrpar(255)             // 0=transparent, 256=deckend
end sub

// Weisse Farbe waehlen
sub white
    call wrcmd(1)               // [1] COLOR r g b alpha
    call wrpar(255)
    call wrpar(255)
    call wrpar(255)
    call wrpar(255)             // 0=transparent, 256=deckend
end sub

// Plot formulieren (wenn keine externe Datei vorliegt)
sub create
    dim shorty as short
    dim home,file as string
    home="/storage/emulated/0/" // allg. Android-Ordner
    file="gforth/home/plot.dat" // Datei im Gforth-Ordner
    open #1,home+file,"w"       // schreibend zugreifen

    call black
    call wrcmd(2)               // [2] CLEAR

    call white                  // weiss

    call show                   // Pause

    call wrcmd(4)               // [4] RECT x y w h f
    call w2par(400,300)
    call w2par(800,600)
    call wrpar(0)

    call wrcmd(3)               // [3] LINE x1 y1 x2 y2
    call w2par(400,300)
    call w2par(1200,900)

    call wrcmd(3)               // [3] LINE x1 y1 x2 y2
    call w2par(1200,300)
    call w2par(400,900)

    call show                   // Pause

    call wrcmd(5)               // [5] OVAL x y w h f
    call w2par(400,300)
    call w2par(800,600)
    call wrpar(0)

    call show                   // Pause

    call wrcmd(6)               // [6] CIRCLE x y r f
    call w2par(800,500)
    call wrpar(200)
    call wrpar(0)

    call show                   // Pause

    call wrcmd(12)              // [12] PIE x y w h b a f
    call w2par(627,330)
    call w2par(350,350)
    call w2par(220,100)
    call wrpar(1)

    call grey                   // grau

    call wrpar(11)              // [11] ARC x y w h b a f
    call w2par(400,300)
    call w2par(800,600)
    call w2par(45,90)
    call wrpar(1)

    call show                   // Pause

    call white                  // weiss

    call wrcmd(17)              // [17] RNDRECT x y w h r f
    call w2par(350,250)
    call w2par(900,700)
    call wrpar(50)
    call wrpar(0)

    call show                   // Pause

    call wrcmd(19)              // [19] SAVE

    call wrcmd(16)              // [16] ROTATE d
    call wrpar(-10)

    dim fs,ft as float
    dim i as integer

    call wrcmd(7)               // [7] PLOT ... [0]
    for i=0 to 980
        fs=sin(float(i)/float(30))
        ft=fs*float(100)+float(770)
        call w2par(integer(i)+188,integer(ft))
    end for
    call wrcmd(0)               // [0] ABSCHLUSS 

    call wrcmd(20)              // [20] RESTORE

    call show                   // Pause

    call grey                   // grau

    call wrcmd(18)              // [18] TRIANG x1 y1 x2 y2 x3 y3 f
    call w2par(1300,900)
    call w2par(1700,500)
    call w2par(1400,300)
    call wrpar(1)

    call show                   // Pause

    call white                  // weiss

    call wrcmd(18)              // [18] TRIANG x1 y1 x2 y2 x3 y3 f
    call w2par(1500,650)
    call w2par(1600,460)
    call w2par(1400,460)
    call wrpar(1)

    call show                   // Pause

    call wrcmd(15)              // [15] ORIGIN x y
    call w2par(190,940)

    call wrcmd(16)              // [16] ROTATE a
    call wrpar(-90)

    call wrcmd(13)              // [13] FONT typ h
    call w2par(3,70)            // 3=Serif

    call wrcmd(14)              // [14] TEXT x y abcd [0]
    call w2par(135,100)
    call wrpar(256*asc("H")+asc("e"))
    call wrpar(256*asc("l")+asc("l"))
    call wrpar(256*asc("o")+asc(" "))
    call wrpar(256*asc("W")+asc("o"))
    call wrpar(256*asc("r")+asc("l"))
    call wrpar(256*asc("d")+asc(" "))
    call wrcmd(0)               // [0] ABSCHLUSS

    call wrcmd(9)               // [9] SHOW

    call wrcmd(8)               // [8] PAUSE s
    call wrpar(100)             // 100 Sekunden

    call wrcmd(10)              // [10] END

    close #1                    // Datei schliessen
end sub


/* ######################## */
/* Hier beginnt der Plotter */
/* ######################## */

function int(x as short) as short
    int=integer(x) & 0xFF
end function

// ######## COLOR - Farbe waehlen
sub gocolor                     // [1] COLOR r g b alpha
    print "color"
    dim r,g,b,alpha as short
    get #1,r                    // rot
    get #1,g                    // gruen
    get #1,b                    // blau
    get #1,alpha                // alpha (255=deckend)
    setcolor r,g,b,alpha
end sub

// ######## CLEAR - Mit gewaehlter Farbe Screen fuellen 
sub goclear                     // [2] CLEAR
    print "clear"
    cls
end sub

// ######## LINE - Linie zeichnen
sub goline                      // [3] LINE x1 y1 x2 y2 
    print "line"
    dim x1,y1,x2,y2 as short
    get #1,x1                   // x1 y1
    get #1,y1
    get #1,x2                   // x2 y2
    get #1,y2
    drawline x1,y1,x2,y2
end sub

// ######## RECT - Rechteck zeichnen (hohl oder voll)
sub gorect                      // [4] RECT x y w h f
    print "rect"
    dim x,y,w,h,f as short
    get #1,x                    // x y links oben
    get #1,y
    get #1,w                    // Breite
    get #1,h                    // Hoehe
    get #1,f                    // fill (0=hohl, 1=voll)
    if integer(f)=0 then
        drawrect x,y,w,h
    else
        fillrect x,y,w,h
    end if
end sub

// ######## OVAL - Ellipse zeichnen (hohl oder voll)
sub gooval                      // [5] OVAL x y w h f
    print "oval"
    dim x,y,w,h,f as short
    get #1,x                    // x y links oben
    get #1,y
    get #1,w                    // Breite Rechteck
    get #1,h                    // Hoehe Rechteck
    get #1,f                    // fill (0=hohl, 1=voll)
    if integer(f)=0 then
        drawoval x,y,w,h
    else
        filloval x,y,w,h
    end if
    if integer(f)=0 then
        drawrect x,y,w,h
    else
        fillrect x,y,w,h
    end if
end sub

// ######## CIRCLE - Kreis zeichnen (hohl oder voll)
sub gocircle                    // [6] CIRCLE x y r f
    print "circle"
    dim x,y,r,f as short
    get #1,x                    // x y links oben
    get #1,y
    get #1,r                    // Radius
    get #1,f                    // fill (0=hohl, 1=voll) 
    if integer(f)=0 then
        drawcircle x,y,r
    else
        fillcircle x,y,r
    end if
end sub

// ######## PLOT - Eine Folge von Punkten setzen
sub goplot                      // [7] PLOT x1 y1 x2 y2 ... [0]
    print "plot"
    dim x,y as short
    dim endflag as boolean
    endflag=false
    while endflag=false
        get #1,x
        if x=short(0xAA00) then // [0] Ende-Marke
            endflag=true
        else
            get #1,y
            plot x,y
        end if
    end while
end sub

// ######## PAUSE - Eine Anzahl Sekunden warten
sub gopause                     // [8] PAUSE s
    print "pause"
    dim s as short
    get #1,s
    sleep integer(s)*1000       // mal 1000 Millisekunden
end sub

// ######## SHOW - Plot sichtbar machen
sub goshow                      // [9] SHOW
    print "show"
    repaint
end sub

// ######## END - Plotdatei-Ende
sub goend                       // [10] END
    print "end"
end sub

// ######## ARC - Ellipsenbogen zeichnen (hohl oder voll)
sub goarc                       // [11] ARC x y w h b a f
    print "arc"
    dim x,y,w,h,b,a,f as short
    get #1,x                    // x y links oben
    get #1,y
    get #1,w                    // Rechteck-Breite
    get #1,h                    // Rechteck-Hoehe
    get #1,b                    // Winkel-Beginn
    get #1,a                    // Winkel-Weite
    get #1,f                    // fill (0=hohl, 1=voll)
    if integer(f)=0 then
        drawarc x,y,w,h,b,a
    else
        fillarc x,y,w,h,b,a
    end if
end sub

// ######## PIE - Tortenstueck zeichnen (hohl oder voll)
sub gopie                       // [12] PIE x y w h b a f
    print "pie"
    dim x,y,w,h,b,a,f as short
    get #1,x                    // x y links oben
    get #1,y
    get #1,w                    // Rechteck-Breite
    get #1,h                    // Rechteck-Hoehe
    get #1,b                    // Winkel-Beginn
    get #1,a                    // Winkel-Weite
    get #1,f                    // fill (0=hohl, 1=voll)
    if integer(f)=0 then
        drawpie x,y,w,h,b,a
    else
        fillpie x,y,w,h,b,a
    end if
end sub

// ######## FONT - Zeichensatz und Hoehe waehlen
sub gofont                      // [13] FONT typ h
    print "font"
    dim typ,h as short
    get #1,typ                  // Typ: 1=Mono, 2=SansSerif, 3=Serif
    get #1,h                    // Hoehe in Pixeln
    if integer(typ)=1 then
        setfont "MONOSPACE",h
    elseif integer(typ)=2 then
        setfont "SANS-SERIF",h
    elseif integer(typ)=3 then
        setfont "SERIF",h
    end if
end sub

// ######## TEXT - Zeichenkette einfuegen
// Der Text besteht aus einer geraden Anzahl Bytes,
// ggf. ist das letzte Zeichen ein Leerzeichen.
//      +-------+-------+-------+-------+-------+-------+
//      | $AA0D |  <x>  |  <y>  | a : b | c : _ | $AA00 |
//      +-------+-------+-------+-------+-------+-------+
// Erstes Zeichen (hier "a") im hoeherwertigen Byte ("big endian").
 
sub gotext                      // [14] TEXT x y a b c ... [0]
    print "text"
    dim item as short
    dim st$,lo$,hi$ as string
    dim x,y  as short
    get #1,x
    get #1,y
    st$=""
    get #1,item
    while item<>short(0xAA00)   // [0] Ende-Marke
        lo$=chr$(integer(item))
        hi$=chr$(integer(item)/256)
        st$=st$+hi$+lo$
        get #1,item
    end while
    drawstring st$,x,y
end sub

// ######## ORIGIN - Bezugsort fuer Schieben und Drehen waehlen
sub goorigin                    // [15] ORIGIN x y
    dim x,y as short
    get #1,x
    get #1,y
    translate x,y
end sub

// ######## ROTATE - Nachfolgenden Plot drehen
// (ggf. mit SAVE und RESTORE alten Zustand wiederherstellen)
sub gorotate                    // [16] ROTATE a
    print "rotate"
    dim a as short
    get #1,a
    rotate a
end sub

// ######## RNDRECT - Rechteck mit runden Ecken zeichnen (hohl oder voll)
sub gorndrect                   // [17] RNDRECT x y w h r f
    print "rndrect"
    dim x,y,w,h,r,f as short
    get #1,x                    // x y links oben
    get #1,y
    get #1,w                    // Breite
    get #1,h                    // Hoehe
    get #1,r                    // Ecken-Radius
    get #1,f                    // fill (0=hohl, 1=voll)
    if integer(f)=0 then
        drawroundrect x,y,w,h,r,r
    else
        fillroundrect x,y,w,h,r,r
    end if
end sub

// ######## TRIANG - Dreieck zeichnen (hohl oder voll)
// (In Mobile Basic ist keine Triangle-Funktion vorhanden.)
//
// Das hohle Dreieck wird aus drei Linien gebildet.
// Das volle Dreieck muss zeilenweise gefuellt werden.
// Dazu ist folgende Anordnung erforderlich:
//              Punkt A (x1,y1) zu oberst, Punkt C (x3,y3) zu unterst.
//        A     Punkt B = x2,y2 stellt den Knickpunkt dar. 
//       / \    
//      /   B   Es werden Zeilen von A bis C gezeichnet,
//     /  /     jeweils von Linie A-C bis A-B bzw. B-C,
//    / /       wobei Anfangs- und Endpunkt ausgespart bleiben.
//   //         
//  C           B darf auch auf der Hoehe von A oder C liegen.

sub gotriang                    // [18] TRIANG x1 y1 x2 y2 x3 y3 f
    print "triang"
    dim x1,y1,x2,y2,x3,y3,f as short
    dim x0,y0,yl,ys as short
    dim i,j as integer
    dim xx1,xx2,yy as float
    get #1,x1                   // x1 y1
    get #1,y1
    get #1,x2                   // x2 y2
    get #1,y2
    get #1,x3                   // x3 y3
    get #1,y3
    get #1,f                    // fill (0=hohl, 1=voll)

// hohles Dreieck
    drawline x1,y1,x2,y2        // Dreieck zeichnen
    drawline x2,y2,x3,y3
    drawline x3,y3,x1,y1

    if integer(f)>0 then
// gefuelltes Dreieck
// Passende Reihenfolge der Punkte A B C durch Tausch erzwingen
//       z.B. y1=300, y2=200, y3=100
// 1. Tausch: y1=200, y2=300, y3=100
// 2. Tausch: y1=200, y2=100, y3=300
// 3. Tausch: y1=100, y2=200, y3=300
        if y1>y2 then           // y1 <= y2
            x0=x1
            y0=y1
            x1=x2
            y1=y2
            x2=x0
            y2=y0
        end if
        if y2>y3 then           // y2 <= y3
            x0=x2
            y0=y2
            x2=x3
            y2=y3
            x3=x0
            y3=y0
        end if
        if y1>y2 then           // y1 <= y2
            x0=x1
            y0=y1
            x1=x2
            y1=y2
            x2=x0
            y2=y0
        end if

// Dreieck fuellen
        yl=y3-y1
        ys=y2-y1
        for i=1 to integer(yl)-1
            j=i-integer(ys)
            xx1=float(x1)+float(x3-x1)*float(i)/float(y3-y1)
            if i<(y2-y1) then
                xx2=float(x1)+float(x2-x1)*float(i)/float(y2-y1)
            else
                xx2=float(x2)+float(x3-x2)*float(j)/float(y3-y2)
            end if
            yy=float(y1)+float(i)
            drawline xx1,yy,xx2,yy
        end for
    end if
end sub

// ######## SAVE - Grafikausrichtung speichern 
sub gosave                      // [19] SAVE
    gsave
end sub

// ######## RESTORE - Grafikausrichtung wiederherstellen
sub gorestore                   // [20] RESTORE
    grestore
end sub


/* ############# */
/* HAUPTSCHLEIFE */
/* ############# */

sub main
    dim home,file as string
    dim ding as short           // Kommando-Kennnummer
    dim item,cmd as integer
    dim eflag as boolean        // Dateiende-Flag
    dim intern as boolean

intern = 1          // 0 = extern, 1 = intern erzeugte Plotdatei

    graphics

    if intern=true then
        call create             // intern erzeugte Plotdatei
    end if

    home="/storage/emulated/0/" // allg. Android-Ordner
    file="gforth/home/plot.dat" // Datei im Gforth-Ordner
    open #1,home+file,"r"       // lesend zugreifen

    repeat
        repeat
            get #1,ding
            item=integer(ding)
        until item<0xAA00
        cmd=item & 0xFF
        if cmd=1 then
            call gocolor
        elseif cmd=2 then
            call goclear
        elseif cmd=3 then
            call goline
        elseif cmd=4 then
            call gorect
        elseif cmd=5 then
            call gooval
        elseif cmd=6 then
            call gocircle
        elseif cmd=7 then
            call goplot
        elseif cmd=8 then
            call gopause
        elseif cmd=9 then
            call goshow
        elseif cmd=10 then
            print "end"
            eflag=true
        elseif cmd=11 then
            call goarc
        elseif cmd=12 then
            call gopie
        elseif cmd=13 then
            call gofont
        elseif cmd=14 then
            call gotext
        elseif cmd=15 then
            call goorigin
        elseif cmd=16 then
            call gorotate
        elseif cmd=17 then
            call gorndrect
        elseif cmd=18 then
            call gotriang
        elseif cmd=19 then
            call gosave
        elseif cmd=20 then
            call gorestore
        end if
    until eflag=true

    close #1
end sub
